define([], function() {
    var t = {
        init: function() {
            'use strict';

            var document = typeof window !== 'undefined' && typeof window.document !== 'undefined' ? window.document : {};
            var keyboardAllowed = typeof Element !== 'undefined' && 'ALLOW_KEYBOARD_INPUT' in Element;

            var fn = (function () {
                var val;

                var fnMap = [
                    [
                        'requestFullscreen',
                        'exitFullscreen',
                        'fullscreenElement',
                        'fullscreenEnabled',
                        'fullscreenchange',
                        'fullscreenerror'
                    ],
                    // New WebKit
                    [
                        'webkitRequestFullscreen',
                        'webkitExitFullscreen',
                        'webkitFullscreenElement',
                        'webkitFullscreenEnabled',
                        'webkitfullscreenchange',
                        'webkitfullscreenerror'

                    ],
                    // Old WebKit (Safari 5.1)
                    [
                        'webkitRequestFullScreen',
                        'webkitCancelFullScreen',
                        'webkitCurrentFullScreenElement',
                        'webkitCancelFullScreen',
                        'webkitfullscreenchange',
                        'webkitfullscreenerror'

                    ],
                    [
                        'mozRequestFullScreen',
                        'mozCancelFullScreen',
                        'mozFullScreenElement',
                        'mozFullScreenEnabled',
                        'mozfullscreenchange',
                        'mozfullscreenerror'
                    ],
                    [
                        'msRequestFullscreen',
                        'msExitFullscreen',
                        'msFullscreenElement',
                        'msFullscreenEnabled',
                        'MSFullscreenChange',
                        'MSFullscreenError'
                    ]
                ];

                var i = 0;
                var l = fnMap.length;
                var ret = {};

                for (; i < l; i++) {
                    val = fnMap[i];
                    if (val && val[1] in document) {
                        for (i = 0; i < val.length; i++) {
                            ret[fnMap[0][i]] = val[i];
                        }
                        return ret;
                    }
                }

                return false;
            })();

            var eventNameMap = {
                change: fn.fullscreenchange,
                error: fn.fullscreenerror
            };

            var screenfull = {
                request: function (elem) {
                    return new Promise(function (resolve) {
                        var request = fn.requestFullscreen;

                        var onFullScreenEntered = function () {
                            this.off('change', onFullScreenEntered);
                            resolve();
                        }.bind(this);

                        elem = elem || document.documentElement;

                        // Work around Safari 5.1 bug: reports support for
                        // keyboard in fullscreen even though it doesn't.
                        // Browser sniffing, since the alternative with
                        // setTimeout is even worse.
                        if (/ Version\/5\.1(?:\.\d+)? Safari\//.test(navigator.userAgent)) {
                            elem[request]();
                        } else {
                            elem[request](keyboardAllowed ? Element.ALLOW_KEYBOARD_INPUT : {});
                        }

                        this.on('change', onFullScreenEntered);
                    }.bind(this));
                },
                exit: function () {
                    return new Promise(function (resolve) {
                        var onFullScreenExit = function () {
                            this.off('change', onFullScreenExit);
                            resolve();
                        }.bind(this);

                        document[fn.exitFullscreen]();

                        this.on('change', onFullScreenExit);
                    }.bind(this));
                },
                toggle: function (elem) {
                    return this.isFullscreen ? this.exit() : this.request(elem);
                },
                onchange: function (callback) {
                    this.on('change', callback);
                },
                onerror: function (callback) {
                    this.on('error', callback);
                },
                on: function (event, callback) {
                    var eventName = eventNameMap[event];
                    if (eventName) {
                        document.addEventListener(eventName, callback, false);
                    }
                },
                off: function (event, callback) {
                    var eventName = eventNameMap[event];
                    if (eventName) {
                        document.removeEventListener(eventName, callback, false);
                    }
                },
                raw: fn
            };

            if (!fn) {
                return false;
            }

            Object.defineProperties(screenfull, {
                isFullscreen: {
                    get: function () {
                        return Boolean(document[fn.fullscreenElement]);
                    }
                },
                element: {
                    enumerable: true,
                    get: function () {
                        return document[fn.fullscreenElement];
                    }
                },
                enabled: {
                    enumerable: true,
                    get: function () {
                        // Coerce to boolean in case of old WebKit
                        return Boolean(document[fn.fullscreenEnabled]);
                    }
                }
            });
            return screenfull;
        }
    };
    return t;
});